%% Fitting a transition model to the data: soil carbonat stable C isotopes
%
% by Martin H. Trauth, 4 December 2020
% http://mres.uni-potsdam.de

%%
% Levin, N.E. (2015) Environment and climate of early human evolution.
% Annu. Rev. Earth Planet. Sci., 43, 405–429.
% https://doi.org/10.1146/annurev-earth060614-105310.
clear, clc, close all
dataraw = load('data_levin_2015.txt');
dataraw(:,1) = -dataraw(:,1);
%dataraw = dataraw(dataraw(:,1)>=-4 & dataraw(:,1)<-1.0,:);
data.Age  = dataraw(:,1);
data.d13C = dataraw(:,2);

%%
% Nonlinear least squares fit of a sigmoid function
fo = fitoptions('Method','NonlinearLeastSquares',...
    'StartPoint',[0 0 -2.8 1],...
    'Lower',[-20 -10 -5  0],...
    'Upper',[ 20  10  5 10]);
ft = fittype(@(a,b,c,d,x) a+b*(1./(1+exp(-d*(x-c)))),...
    'Coeff',{'a','b','c','d'},...
    'Options',fo);

%%
t = -3.5 : 0.1 : -1;
[f,fgof,fout] = fit(data.Age,data.d13C,ft)
y = f.a+f.b*(1./(1+exp(-f.d*(t-f.c))));

%%
rft = load('rampfit.dat');
rs = 'c = -1.570 (-1.778,-1.362)';

%%
xl = ['Age (Ma)'];
yl = ['\delta^{13}C_{sc} (',char(8240),')'];

%%
figure('Position',[50 1000 800 250])
axes('Position',[0.1 0.2 0.8 0.7],...
    'XLim',[-16 0],...
    'YLim',[-15 5],...
    'FontSize',12,...
    'Box','On'), hold on
p = plot(f,data.Age,data.d13C);
p(1).Color = [0.1 0.3 0.6];
p(1).Marker = 'Square';
p(1).MarkerSize = 6;
p(1).MarkerFaceColor = [0.1 0.3 0.6];
p(1).MarkerEdgeColor = [0.1 0.3 0.6];
p(2).Color = [0.7 0.3 0.1];
p(2).LineWidth = 1.5;
p(2).LineStyle = '-';

line(-rft(:,1),rft(:,3),...                % Ramp Fit
    'Color',[0.7 0.3 0.1],...
    'LineWidth',1.5,...
    'LineStyle','-.')

lgd = legend('Data Points','Sigmoid Fit','Ramp Fit');
lgd.Location = 'northwest';
lgd.Box = 'Off';
lgd.FontSize = 12;
lgd.AutoUpdate = 'Off';
xlabel(xl,...
    'FontSize',12)
ylabel(yl,...
    'FontSize',12)

line([-1.47 -1.47],[1 3],...              % Inflection Sigmoid Fit
    'LineWidth',1.5,...
    'Color',[0.7 0.3 0.1])
text(-1.47,4,'-1.47 Ma',...
    'FontSize',18,...
    'Color',[0.7 0.3 0.1],...
    'HorizontalAlignment','Right')

line([-1.57 -1.57],[-13 -8],...              % Inflection Ramp Fit
    'LineWidth',1.5,...
    'LineStyle','-.',...
    'Color',[0.7 0.3 0.1])
text(-1.57,-14,'-1.57 Ma',...
    'FontSize',18,...
    'Color',[0.7 0.3 0.1],...
    'HorizontalAlignment','Left')

print -dpng -r300 commands_levin2015_sigmoid_ramp_vs2_FULL.png




